<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Server;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class ServerController extends Controller
{
    /**
     * Display a listing of the servers.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $servers = Server::all();
        return view('admin.servers.index', compact('servers'));
    }

    /**
     * Show the form for creating a new server.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('admin.servers.create');
    }

    /**
     * Store a newly created server in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        // Debug the incoming request data
        Log::info('Creating server with data:', $request->all());

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'ip' => 'required|string|max:45',
            'port' => 'required|numeric|min:1|max:65535',
            'app_id' => 'nullable|string|max:20',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            Log::warning('Server validation failed:', $validator->errors()->toArray());
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $server = new Server();
            $server->name = $request->name;
            $server->ip = $request->ip;
            $server->port = $request->port;
            $server->app_id = $request->app_id ?: '4000';
            $server->is_active = $request->has('is_active');
            $server->api_key = $server->generateApiKey();
            $server->save();

            Log::info('Server created successfully:', ['id' => $server->id, 'name' => $server->name]);
            
            return redirect()->route('admin.servers.index')
                ->with('success', 'Server created successfully.');
        } catch (\Exception $e) {
            Log::error('Error creating server:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->with('error', 'Error creating server: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified server.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $server = Server::findOrFail($id);
        
        // Try to query server info from Steam API
        $serverInfo = $server->queryServerInfo();
        
        return view('admin.servers.show', compact('server', 'serverInfo'));
    }

    /**
     * Show the form for editing the specified server.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $server = Server::findOrFail($id);
        return view('admin.servers.edit', compact('server'));
    }

    /**
     * Update the specified server in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id)
    {
        // Debug the incoming request data
        Log::info('Updating server #' . $id . ' with data:', $request->all());

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'ip' => 'required|string|max:45',
            'port' => 'required|numeric|min:1|max:65535',
            'app_id' => 'nullable|string|max:20',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            Log::warning('Server update validation failed:', $validator->errors()->toArray());
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $server = Server::findOrFail($id);
            
            // Store original values for logging
            $original = [
                'name' => $server->name,
                'ip' => $server->ip,
                'port' => $server->port,
                'app_id' => $server->app_id,
                'is_active' => $server->is_active
            ];
            
            $server->name = $request->name;
            $server->ip = $request->ip;
            $server->port = $request->port;
            $server->app_id = $request->app_id ?: '4000';
            $server->is_active = $request->has('is_active');
            
            // Check if any fields were actually changed
            $changes = array_diff_assoc([
                'name' => $server->name,
                'ip' => $server->ip,
                'port' => $server->port,
                'app_id' => $server->app_id,
                'is_active' => $server->is_active
            ], $original);
            
            if (!empty($changes)) {
                Log::info('Server changes detected:', $changes);
                $server->save();
                Log::info('Server updated successfully:', ['id' => $server->id]);
                
                return redirect()->route('admin.servers.show', $server->id)
                    ->with('success', 'Server updated successfully.');
            } else {
                Log::info('No changes detected for server:', ['id' => $server->id]);
                
                return redirect()->route('admin.servers.show', $server->id)
                    ->with('info', 'No changes were made to the server.');
            }
        } catch (\Exception $e) {
            Log::error('Error updating server:', [
                'id' => $id,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->with('error', 'Error updating server: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified server from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        try {
            $server = Server::findOrFail($id);
            Log::info('Deleting server:', ['id' => $id, 'name' => $server->name]);
            $server->delete();

            return redirect()->route('admin.servers.index')
                ->with('success', 'Server deleted successfully.');
        } catch (\Exception $e) {
            Log::error('Error deleting server:', [
                'id' => $id,
                'message' => $e->getMessage()
            ]);
            
            return redirect()->back()
                ->with('error', 'Error deleting server: ' . $e->getMessage());
        }
    }

    /**
     * Reset the API key for the specified server.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function resetApiKey($id)
    {
        try {
            $server = Server::findOrFail($id);
            $apiKey = $server->resetApiKey();
            
            Log::info('API key reset for server:', ['id' => $id, 'name' => $server->name]);

            return redirect()->route('admin.servers.show', $server->id)
                ->with('success', 'API key has been reset successfully.')
                ->with('apiKey', $apiKey);
        } catch (\Exception $e) {
            Log::error('Error resetting API key:', [
                'id' => $id,
                'message' => $e->getMessage()
            ]);
            
            return redirect()->back()
                ->with('error', 'Error resetting API key: ' . $e->getMessage());
        }
    }

    /**
     * Refresh server information from Steam API.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function refreshInfo($id)
    {
        try {
            $server = Server::findOrFail($id);
            $serverInfo = $server->queryServerInfo();

            if ($serverInfo) {
                Log::info('Server info refreshed successfully:', ['id' => $id, 'name' => $server->name]);
                
                return redirect()->route('admin.servers.show', $server->id)
                    ->with('success', 'Server information refreshed successfully.')
                    ->with('serverInfo', $serverInfo);
            } else {
                Log::warning('Failed to retrieve server info:', ['id' => $id, 'name' => $server->name]);
                
                return redirect()->route('admin.servers.show', $server->id)
                    ->with('error', 'Failed to retrieve server information. Please check the server details and Steam API key.');
            }
        } catch (\Exception $e) {
            Log::error('Error refreshing server info:', [
                'id' => $id,
                'message' => $e->getMessage()
            ]);
            
            return redirect()->back()
                ->with('error', 'Error refreshing server info: ' . $e->getMessage());
        }
    }
}